<?php

/**
 * Exception to raise in Qb framework.
 */
class QbException extends Exception {}


/**
 * Simple application registry
 */
class QbRegistry
{
    protected static $_properties = array();

    public static function get($alias, $default = NULL)
    {
        if (isset(self::$_properties[$alias])) {
            return self::$_properties[$alias];
        }
        return $default;
    }

    public static function set($alias, $value)
    {
    	self::$_properties[$alias] = $value;
    }
}


/**
 * Application configuration and component storage
 */
class Qb extends QbRegistry
{
    public static function useConfig($config)
    {
    	self::_init();
    	if (!is_array($config)) {
    		throw new QbException('Wrong core config');
    	}
    	if (isset($config['components'])) {
    		self::$_components = array_merge(self::$_components, $config['components']);
    		unset($config['components']);
    	}
    	if (isset($config['classes'])) {
    		self::$_classes = array_merge(self::$_classes, $config['classes']);
    		unset($config['classes']);
    	}
    	self::$_properties = array_merge(self::$_properties, $config);
    }

	/**
	 * c - abbreviation of "component"
	 */
    public static function c($alias, $interfaces = '')
    {
    	$ret = self::get($alias);
		// First time call?
    	if (is_null($ret) && isset(self::$_components[$alias])) {
    		$params = self::$_components[$alias];
			$type = $params['class'];
			unset($params['class']);
			if (!class_exists($type, TRUE)) {
				throw new QbException('Undefined class "' . $type . '"');
			}
			foreach ($params as $key => $value) {
				if ($key{0} == '@') {
					unset($params[$key]);
					$params[substr($key, 1)] = self::get($value);
				}
			}
			self::set($alias, $ret = new $type($params));
		}
		if (is_null($ret)) {
			throw new QbException('Undefined component "' . $alias . '"');
		}
		// Have to meet interfaces?
    	if (!empty($interfaces)) {
			if (is_string($interfaces) && strpos($interfaces, ',') !== FALSE) {
				$interfaces = array_map('trim', explode(',', $interfaces));
			}
			if (is_array($interfaces)) {
				foreach ($interfaces as $interface) {
					if (!($ret instanceOf $interface)) {
						throw new QbException($alias . ' is not '. $interface);
					}
				}
			} else if (!($ret instanceOf $interfaces)) {
				throw new QbException($alias . ' is not '. $interfaces);
			}
    	}
    	return $ret;
    }

	private static
		$_initiated  = FALSE,
		$_components = array(),
		$_classes    = array();

	private static function _handleClass($className)
	{
		$c = strtolower($className);
		if (isset(self::$_classes[$c])) {
			$fileName = self::$_classes[$c];
		} else {
			foreach (self::$_classes as $head => $dir) {
				$p = strpos($head, '*'); // yes, I know about FALSE
				if ($p > 0 && strncmp($head, $c, $p) == 0) {
					$fileName = $dir . $c . '.php';
					break;
				}
			}
		}
		if (empty($fileName)) {
			$fileName = self::$_classes['*'] . $c . '.php';
		}
		if (file_exists($fileName)) {
			include $fileName;
		}
	}

    private static function _init()
    {
    	if (self::$_initiated) {
    		return;
    	}
    	self::set('start', microtime(TRUE));
		spl_autoload_register(array('QB', '_handleClass'));
		self::$_initiated = TRUE;
	}

}


/**
 * Application error and profile log
 */
class QbLog
{
	private static
		$_log = array();

    const
    	EMERG   = 0,  // Emergency: system is unusable
    	ALERT   = 1,  // Alert: action must be taken immediately
    	CRIT    = 2,  // Critical: critical conditions
    	ERR     = 3,  // Error: error conditions
    	WARN    = 4,  // Warning: warning conditions
    	NOTICE  = 5,  // Notice: normal but significant condition
    	INFO    = 6,  // Informational: informational messages
    	DEBUG   = 7;  // Debug: debug messages

	/**
	 * 
	 */
	public static function add($data, $priority = self::DEBUG)
	{
		self::$_log[] = array(microtime(TRUE), $data, $priority);
	}

	public static function get($filter = NULL)
	{
		if (is_null($filter)) {
			$ret = self::$_log;
		} else if (is_array($filter)) {
			$ret = array();
			foreach (self::$_log as $event) {
				if (in_array($event[2], $filter)) {
					$ret[] = $event;
				}
			}
		} else {
			$ret = array();
			foreach (self::$_log as $event) {
				if ($event[2] <= $filter) {
					$ret[] = $event;
				}
			}
		}
		return $ret;
	}
}


/**
 * Base class for all application components
 */
class QbComponent
{
	private
		$_events = array();

	public function __construct(array $params = NULL)
	{
		if (!empty($params)) {
			foreach ($params as $var => $value) {
				if ($var == 'events') {
					foreach ($value as $name => $handler) {
						$this->addEvent($name, $handler);
					}
				} else {
					$this->$var = $value;
				}
			}
		}
	}

	public function addEvent($name, $handler)
	{
		$name = strtolower($name);
		$this->_events[$name][] = $handler;
		return $this;
	}

	function notify()
	{
		$ret = NULL;
		$args = func_get_args();
		$name = strtolower(array_shift($args));
		if (!empty($this->_events[$name])) {
			foreach ($this->_events[$name] as $handler) {
				$tmp = call_user_func_array($handler, $args);
				if (!is_null($tmp)) {
					$ret = $tmp;
				}
			}
		}
		return $ret;
	}

	function notifyUntil()
	{
		$ret = NULL;
		$args = func_get_args();
		$name = strtolower(array_shift($args));
		if (!empty($this->_events[$name])) {
			foreach ($this->_events[$name] as $handler) {
				if(!is_null($ret = call_user_func_array($handler, $args))) {
					break;
				}
			}
		}
		return $ret;
	}

	public function buildString($text)
	{
		$parts = preg_split('/(\{\$.+\})/U', $text, -1, PREG_SPLIT_DELIM_CAPTURE | PREG_SPLIT_NO_EMPTY);
		$ret = '';
		foreach ($parts as $sub) {
			if (substr($sub, 0, 2) == '{$' && substr($sub, -1, 1) == '}') {
				$sub = substr($sub, 2, -1);
				if (($p = strpos($sub, ':')) === FALSE) {
					$sub = $this->$sub;
				} else {
    				list($from, $to) = explode(',', substr($sub, $p + 1));
					$sub = substr($sub, 0, $p);
					$sub = substr($this->$sub, $from, $to);
				}
			}
			$ret .= $sub;
		}
		return $ret;
	}
}
