<?php
/**
 * Bicycle Libraries
 *
 * Set of libraries wich can help you
 *
 * @package   Bicycle Libraries
 * @author    Konstantin.Myakshin
 * @copyright Copyright (c) 2009-2010, Brouzie, Inc.
 * @license   LGPL
 * @link      http://brouzie.com/projects/bicycle-libraries/
 * @link      http://code.google.com/p/bicycle-libraries/
 * @filesource
 */

class BL_Deploy
{
	protected $settings = array();

	/**
	 *
	 * @var BL_Console
	 */
	protected $c;

	/**
	 *
	 * @var BL_Deploy_ChangesAbstract
	 */
	protected $changer;

	public function __construct($settings)
	{
		$this->settings = $settings;
		$this->c = BL_Console::init();
	}

	protected function getLockFilename($taskName)
	{
		return $this->settings['locksDir'] . $taskName . '.lock';
	}

	protected function getLastUpdateRevision($taskName)
	{
		return file_exists($file = $this->getLockFilename($taskName)) ? file_get_contents($file) : 0;
	}

	protected function setLastUpdateRevision($taskName, $rev)
	{
		file_put_contents($this->getLockFilename($taskName), $rev);
	}

	protected function writeComment($text)
	{
		$this->c->line($this->c->format($text, BL_Console::STYLE_COMMENT));
	}
	
	protected function writeInfo($text)
	{
		$this->c->line('  ' . $text);
	}

	protected function prepareBatch($taskName, $rev = NULL)
	{
		$this->c->line();
		$this->writeComment('Prepearing batch "' . $taskName . '"');

		$this->changer = BL_Deploy_Changes::init($this->settings['repository']);

		$rev || $rev = $this->changer->getRevision();
		$revServer = $this->getLastUpdateRevision($taskName);

		$batch = array(
			'rev'   => $rev,       // revision, to what need update
			'srev'  => $revServer, // current revision on server
			'files' => array()
		);

		$this->writeInfo('Current rev on local repo  ' . $this->changer->getRevision());
		$this->writeInfo('Current rev on server repo ' . $batch['srev']);

		// if server has last revision - no any files need update/remove
		if ($rev == $revServer) {
			$this->writeInfo('Server has last revision. Deploy does not needed');
		} else {
			$batch['files'] = $this->changer->getChanges($revServer, $rev);
			$this->writeInfo('Will update server to rev  ' . $batch['rev']);
			$this->writeInfo('Files count to update u/r  ' . count($batch['files']['upload']) . '/' . count($batch['files']['remove']));

			$this->writeInfo('Export rev ' . $batch['rev']);

			$batch['dir'] = $dir = sys_get_temp_dir() . 'bl_deploy' . rand(1, 9999);
			mkdir($dir, 0777, true);
			$this->changer->export($dir, $batch['rev']);
		}
		
		return $batch;
	}

	protected function runTask($taskName, $batch)
	{
		if (!$batch['files']) {
			return true;
		}
		$this->writeComment('Run batch "' . $taskName . '"');
		$data = $this->settings['tasks'][$taskName];

		try {
			$adapter = BL_Filesystem::init($data['connection'], $data['options']);
		} catch (BL_Filesystem_Exception $e) {
			$this->c->err($e->getMessage());
			return false;
		}

		$errors = array();
		foreach ($batch['files']['upload'] as $file) {
			try {
				$adapter->write($file, file_get_contents($batch['dir'] . '/' . $file));
			} catch (BL_Filesystem_Exception $e) {
				$errors[] = $e->getMessage();
			}
			$this->writeInfo('Uploaded ' . $file);
		}
		foreach ($batch['files']['remove'] as $file) {
			try {
				$adapter->remove($file);
			} catch (BL_Filesystem_Exception $e) {
				$errors[] = $e->getMessage();
			}
			$this->writeInfo('Removed ' . $file);
		}

		if ($errors) {
			foreach ($errors as $error) {
				$this->c->err($error);
			}
		} else {
			$this->setLastUpdateRevision($taskName, $batch['rev']);
		}
	}

	protected function clean($batch)
	{
		if (!isset($batch['dir']))
			return true;

		$this->writeComment('Removing temporary files');
		$fs = new BL_Filesystem_Adapter_Disk();
		$fs->rmDir($batch['dir']);
	}

	public function run($firstRun = true)
	{
		$firstRun && $this->writeInfo("Welcome to bicycle-libraries deployment system\n  Author: Konstantin.Myakshin <brouzie.com>");

		$menu = array(
			'run_all' => 'Run all tasks',
			'run_one' => 'Run custom task',
			'quit'    => 'Quit',
		);

		$action = $this->c->menu($menu, 'run_all', 'Choose action');

		switch ($action) {
			case 'run_all':
				foreach ($this->settings['tasks'] as $taskName => $task) {
					$this->runTask($taskName, $batch = $this->prepareBatch($taskName));
					$this->clean($batch); //TODO:  (    )
				}
				$this->run(false);
				break;

			case 'run_one':
				$taskName = $this->c->prompt('Task name');
				if (isset($this->settings['tasks'][$taskName])) {
					$this->runTask($taskName, $batch = $this->prepareBatch($taskName));
					$this->clean($batch);
				} else {
					$this->c->err('This task does not defined in config file');
				}
				$this->run(false);
				break;

			default:
				$this->c->line('Bye!');
				break;
		}
	}
}
