<?php
/**
 * Bicycle Libraries
 *
 * Set of libraries wich can help you
 *
 * @package   Bicycle Libraries
 * @author    Konstantin.Myakshin
 * @copyright Copyright (c) 2009-2010, Brouzie, Inc.
 * @license   LGPL
 * @link      http://brouzie.com/projects/bicycle-libraries/
 * @link      http://code.google.com/p/bicycle-libraries/
 * @filesource
 */

/**
 * Console class
 * Based on
 *   sfAnsiColorFormatter by Fabien Potencier
 *   cli-tools by James Logsdon http://github.com/jlogsdon/php-cli-tools
 */
class BL_Console
{
	const STYLE_INFO     = 'INFO';
	const STYLE_ERROR    = 'ERROR';
	const STYLE_COMMENT  = 'COMMENT';
	const STYLE_QUESTION = 'QUESTION';

	const RESET_ALL = "\033[0m";

	protected $text = '';

	protected $styles = array(
		self::STYLE_ERROR    => array('bg' => 'red', 'fg' => 'white', 'bold' => true),
		self::STYLE_INFO     => array('fg' => 'green', 'bold' => true),
		self::STYLE_COMMENT  => array('fg' => 'yellow'),
		self::STYLE_QUESTION => array('bg' => 'cyan', 'fg' => 'black', 'bold' => false),
	);
	protected $options    = array('bold' => 1, 'underscore' => 4, 'blink' => 5, 'reverse' => 7, 'conceal' => 8, 'normal' => 22);
	protected $foreground = array('black' => 30, 'red' => 31, 'green' => 32, 'yellow' => 33, 'blue' => 34, 'magenta' => 35, 'cyan' => 36, 'white' => 37);
	protected $background = array('black' => 40, 'red' => 41, 'green' => 42, 'yellow' => 43, 'blue' => 44, 'magenta' => 45, 'cyan' => 46, 'white' => 47);

	/**
	 *
	 * @return BL_Console
	 */
	public static function init()
	{
		return new self;
	}

  /**
   * Sets a new style.
   *
   * @param string $name    The style name
   * @param array  $options An array of options
	 * @return BL_Console
   */
	public function setPredefinedStyle($name, $options = array())
	{
		$this->styles[$name] = $options;
		return $this;
	}

  /**
   * Formats a text according to the given style or parameters.
   *
   * @param  string   $text       The test to style
   * @param  mixed    $parameters An array of options or a style name
   *
   * @return string The styled text
   */
  public function format($text = '', $parameters = array())
  {
    if (!is_array($parameters) && 'NONE' == $parameters) {
      return self::RESET_ALL . $text;
    }

		$reset = false;
    if (!is_array($parameters) && isset($this->styles[$parameters])) {
			$reset = true;
      $parameters =  $this->styles[$parameters]; //TODO: restore previous style
    }

    $codes = array();
    if (isset($parameters['fg'])) {
			//TODO: check exists
      $codes[] = $this->foreground[$parameters['fg']];
    }
    if (isset($parameters['bg'])) {
      $codes[] = $this->background[$parameters['bg']];
    }
    foreach ($this->options as $option => $value) {
      if (isset($parameters[$option]) && $parameters[$option]) {
        $codes[] = $value;
      }
    }

		$start = $reset || $text ? self::RESET_ALL : '';
		$end = $text ? $text . self::RESET_ALL : '';
		return $start . "\033[" . implode(';', $codes) . 'm' . $end;
  }
/*
	// Cursor position
	public function toPos($row = 1, $column = 1)
	{
		echo "\033[{$row};{$column}H";
		return $this;
	}

	public function cursorUp($lines=1)
	{
		echo "\033[{$lines}A";
		return $this;
	}

	public function cursorDown($lines=1)
	{
		echo "\033[{$lines}B";
		return $this;
	}

	public function cursorRight($columns=1)
	{
		echo "\033[{$columns}C";
		return $this;
	}

	public function cursorLeft($columns=1)
	{
		echo "\033[{$columns}D";
		return $this;
	}
*/
	public function options($style = 'default')
	{
		$this->text .= $this->format('', array($style => true));
		return $this;
	}

	public function color($color = 'white')
	{
		$this->text .= $this->format('', array('fg' => $color));
		return $this;
	}

	public function bgColor($color = 'black')
	{
		$this->text .= $this->format('', array('bg' => $color));
		return $this;
	}

	public function restore()
	{
		return $this->out(self::RESET_ALL);
	}

	public function __destruct()
	{
		$this->restore();
	}

	/**
	 * Handles rendering strings. If extra scalar arguments are given after the `$msg`
	 * the string will be rendered with `sprintf`. If the second argument is an `array`
	 * then each key in the array will be the placeholder name. Placeholders are of the
	 * format {:key}.
	 *
	 * @param string   $msg  The message to render.
	 * @param mixed    ...   Either scalar arguments or a single array argument.
	 * @return string  The rendered string.
	 */
	public function render($msg)
	{
		$args = func_get_args();

		// No string replacement is needed
		if (count($args) == 1) {
			return $msg;
		}

		// If the first argument is not an array just pass to sprintf
		if (!is_array($args[1])) {
			return call_user_func_array('sprintf', $args);
		}

		// Here we do named replacement so formatting strings are more understandable
		foreach ($args[1] as $key => $value) {
			$msg = str_replace('{:' . $key . '}', $value, $msg);
		}
		return $msg;
	}

	/**
	 * Shortcut for printing to `STDOUT`. The message and parameters are passed
	 * through `sprintf` before output.
	 *
	 * @param string  $msg  The message to output in `printf` format.
	 * @param mixed   ...   Either scalar arguments or a single array argument.
	 * @return BL_Console
	 */
	public function out($msg)
	{
		$args = func_get_args();
		$this->text .= call_user_func_array(array($this, 'render'), $args);
		fwrite(STDOUT, $this->text);
		$this->text = '';
		return $this;
	}

	/**
	 * Pads `$msg` to the width of the shell before passing to `cli\out`.
	 *
	 * @param string  $msg  The message to pad and pass on.
	 * @param mixed   ...   Either scalar arguments or a single array argument.
	 * @return BL_Console
	 */
	function out_padded($msg)
	{
		$args = func_get_args();
		$msg = call_user_func_array(array($this, 'render'), $args);
		return $this->out(str_pad($msg, 80)); //TODO: get real width
	}

	/**
	 * Prints a message to `STDOUT` with a newline appended. See `out` for
	 * more documentation.
	 * @return BL_Console
	 */
	function line($msg = '')
	{
		// func_get_args is empty if no args are passed even with the default above.
		$args = array_merge(func_get_args(), array(''));
		$args[0] .= PHP_EOL;
		return call_user_func_array(array($this, 'out'), $args);
	}

	/**
	 * Shortcut for printing to `STDERR`. The message and parameters are passed
	 * through `sprintf` before output.
	 *
	 * @param string  $msg  The message to output in `printf` format. With no string,
	 *                      a newline is printed.
	 * @param mixed   ...   Either scalar arguments or a single array argument.
	 * @return BL_Console
	 */
	function err($msg = '')
	{
		// func_get_args is empty if no args are passed even with the default above.
		$args = array_merge(func_get_args(), array(''));
		$args[0] .= PHP_EOL;
		$text = call_user_func_array(array($this, 'render'), $args);
		$text = $this->format($text, self::STYLE_ERROR);
		$this->text .= $text;
		fwrite(STDERR, $this->text);
		$this->text = '';
		return $this;
	}

	/**
	 * Takes input from `STDIN` in the given format. If an end of transmission
	 * character is sent (^D), an exception is thrown.
	 *
	 * @param string  $format  A valid input format. See `fscanf` for documentation.
	 *                         If none is given, all input up to the first newline
	 *                         is accepted.
	 * @return string  The input with whitespace trimmed.
	 * @throws Exception  Thrown if ctrl-D (EOT) is sent as input.
	 */
	function input($format = null)
	{
		if ($format) {
			fscanf(STDIN, $format . PHP_EOL, $line);
		} else {
			$line = fgets(STDIN);
		}

//		if ($line === false) {
//			throw new Exception('Caught ^D during input');
//		}

		return trim($line);
	}

	/**
	 * Displays an input prompt. If no default value is provided the prompt will
	 * continue displaying until input is received.
	 *
	 * @param string  $question  The question to ask the user.
	 * @param string  $default   A default value if the user provides no input.
	 * @param string  $marker    A string to append to the question and default value
	 *                           on display.
	 * @return string  The users input.
	 */
	function prompt($question, $default = false, $marker = ':')
	{
		if ($default && strpos($question, '[') === false) {
			$question .= ' [' . $default . ']';
		}

		while (true) {
			$this
				->out($this->format($question . $marker, self::STYLE_QUESTION))
				->out(' ');
			$line = $this->input();

			if (!empty($line))
				return $line;
			if ($default !== false)
				return $default;
		}
	}

	/**
	 * Presents a user with a multiple choice question, useful for 'yes/no' type
	 * questions (which this function defaults too).
	 *
	 * @param string  $question  The question to ask the user.
	 * @param string  $valid     A string of characters allowed as a response. Case
	 *                           is ignored.
	 * @param string  $default   The default choice. NULL if a default is not allowed.
	 * @return string  The users choice.
	 */
	function choose($question, $choice = 'yn', $default = 'n')
	{
		if (!is_string($choice)) {
			$choice = join('', $choice);
		}

		// Make every choice character lowercase except the default
		$choice = str_ireplace($default, strtoupper($default), strtolower($choice));
		// Seperate each choice with a forward-slash
		$choices = trim(join('/', preg_split('//', $choice)), '/');

		while (true) {
			$text = sprintf('%s [%s]', $question . '?', $choices);
			$line = $this->prompt($text, $default, '');

			if (stripos($choice, $line) !== false) {
				return strtolower($line);
			}
			if (!empty($default)) {
				return strtolower($default);
			}
		}
	}

	/**
	 * Displays an array of strings as a menu where a user can enter a number to
	 * choose an option. The array must be a single dimension with either strings
	 * or objects with a `__toString()` method.
	 *
	 * @param array   $items    The list of items the user can choose from.
	 * @param string  $default  The index of the default item.
	 * @param string  $title    The message displayed to the user when prompted.
	 * @return string  The index of the chosen item.
	 */
	function menu($items, $default = false, $title = 'Choose an item')
	{
		$map = array_values($items);

		if ($default && strpos($title, '[') === false && isset($items[$default])) {
			$title .= ' [' . $items[$default] . ']';
		}

		foreach ($map as $idx => $item) {
			$idx = $this->format(sprintf('%d.', $idx + 1), self::STYLE_INFO);
			$this->line()
				->out('  ')
				->out($idx)
				->out(' ')
				->out((string)$item);
		}
		$this->line();

		while (true) {
			$this
				->out($this->format($title . ':', self::STYLE_QUESTION))
				->out(' ');
			$line = $this->input();

			if (is_numeric($line)) {
				$line--;
				if (isset($map[$line])) {
					return array_search($map[$line], $items);
				}

				if ($line <= 0 || $line >= count($map)) {
					$this->err('Invalid menu selection: out of range');
				}
			} elseif ($line) { // line entered but not numeric
				$this->err('Invalid menu selection: only numbers allowed');
			} elseif ($default) { // empty line
				return $default;
			}
		}
	}
}