<?php
/********************************************************************
 * RBAC/test1.php - Simple Role-Based-Access-Control checker test
 * - meta-data model stored in PHP for simplicity
 * - two levels of authorization identity: tasks-operation
 * - you can change deny-allow order via 'CPermsChecker::byDefault'
 *
 * (c) 2009, artoodetoo
 *
 * All in one file.
 * Demo purposes only.
 ********************************************************************/


// Dummy meta-data model
class CPermsModel
{
/* --- public section -- */

	public function getRoles()
	{
		return array('administrator', 'moderator', 'user', 'guest', 'post owner', 'banned');
	}

	public function getTasks()
	{
		return array(
			'user management' => array('create user', 'modify user', 'delete user'),
			'post management' => array('create post', 'modify post', 'delete post')
		);
	}

	public function getGrants()
	{
		return array(
			array('administrator',	'allow',	'user management'),
			array('administrator',	'allow',	'post management'),
			array('moderator',		'allow',	'post management'),
			array('banned',			'deny',		'user management'),
			array('banned',			'deny',		'post management'),
		//	array('user',			'allow',	'post management'),
			array('user',			'allow',	'create post'),
			array('post owner',		'allow',	'modify post'),
			array('post owner',		'allow',	'delete post'),
		);
	}
}


// Simple permission checker
class CPermsChecker
{
/* --- private section -- */

	private $roles;
	private $tasks;
	private $grants;

	//
	// Check one role for specified action or task
	// returns allow|deny|NULL
	private function findGrant($role, $entity)
	{
		foreach ($this->grants as $grant)
		{
			if ($role != $grant[0])
				continue;

			// Is name found? or
			// Is it task and name is task item?
			if (($entity == $grant[2]) ||
			    (isset($this->tasks[$grant[2]]) && in_array($entity, $this->tasks[$grant[2]])))
				return $grant[1];
		}

		return NULL;
	}

/* --- public section -- */

	const byDefault = 'deny'; // deny all then allow something

	function __construct(CPermsModel $model)
	{
		// Fetch meta-data from database
		$this->roles = $model->getRoles();
		$this->tasks = $model->getTasks();
		$this->grants = $model->getGrants();
	}

	//
	// Can it do the action?
	// returns allow|deny
	// first arg is object having roles or roles themleslf
	// second arg is action or task
	//
	public function can($object, $entity)
	{
		// It is an object_with_roles_property OR array_of_role_ids OR role_id
		if (is_object($object))
			$roles = $object->roles;
		else if (is_array($object))
			$roles = $object;
		else
			$roles = array($object);

		$result = NULL;

		// Test every role...
		foreach ($roles as $role)
		{
			$g = $this->findGrant($role, $entity);
			// ... until first wanted value found
			if (!is_null($g))
			{
				$result = $g;
				if ($g == self::byDefault)
					break;
			}
		}

		if (is_null($result))
			return self::byDefault;

		return $result;
	}

	//
	// Get role_ids array who can|cannot do the action
	// first arg is action or task
	// second arg is deny|allow
	//
	public function whoCan($entity, $value = 'allow')
	{
		$result = array();

		foreach ($this->roles as $role)
		{
			$g = $this->findGrant($role, $entity);
			if ($g == $value || (is_null($g) && $value == self::byDefault))
				$result[] = $role;
		}

		return $result;
	}
}


// Tests:

$model = new CPermsModel;
$p = new CPermsChecker($model);

header('Content-type: text/plain');

echo "By default = ".CPermsChecker::byDefault;

echo "\n\n   role - operation";
echo "\n1. ".$p->can('guest', 'delete post');
echo "\n2. ".$p->can('user', 'delete post');
echo "\n3. ".$p->can('post owner', 'delete post');
echo "\n4. ".$p->can('moderator', 'delete post');

echo "\n\n   role - task";
echo "\n1. ".$p->can('moderator', 'post management');
echo "\n2. ".$p->can('moderator', 'user management');

echo "\n\n   role set - operation";
echo "\n1. ".$p->can(array('user', 'post owner'), 'delete post');
echo "\n2. ".$p->can(array('user', 'banned'), 'delete post');
echo "\n3. ".$p->can(array('user', 'moderator', 'banned'), 'delete post');

echo "\n\n   role set - task";
echo "\n1. ".$p->can(array('user'), 'post management');
echo "\n1. ".$p->can(array('user', 'moderator'), 'post management');
echo "\n2. ".$p->can(array('user', 'moderator', 'banned'), 'post management');

echo "\n\n   who can";
echo "\ncreate post:     ".implode(', ', $p->whoCan('create post'));
echo "\ndelete post:     ".implode(', ', $p->whoCan('delete post'));
echo "\npost management: ".implode(', ', $p->whoCan('post management'));

echo "\n\n   who cannot";
echo "\ncreate post:     ".implode(', ', $p->whoCan('create post', 'deny'));
echo "\ndelete post:     ".implode(', ', $p->whoCan('delete post', 'deny'));
echo "\npost management: ".implode(', ', $p->whoCan('post management', 'deny'));
